/*
 * Copyright (C) Sergey Kolesov 2012-2021 <kolesov@ocean.phys.msu.ru>
 * 
 * ffaultdisp is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * ffaultdisp is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <getopt.h>
#include <errno.h>
#include <unistd.h>
#include <string>

#include "okada.h"
#include "solver.h"
#include "config.h"

#define OPT_PRECISION 1000

using namespace std;
//===============================
    //  Commandline (getopts_long) related stuff
static struct option OptionsLong[] =
{
 {"help", no_argument, 0, 'h'},
 {"output", required_argument, 0, 'o'},
 {"region", required_argument, 0, 'r'},
 {"set", required_argument, 0, 's'},
 {"interval", required_argument, 0, 'i'},
 {"CR", required_argument, 0, 'c'},
 {"fault-pos", required_argument, 0, 'p'},
 {"dynamic", required_argument, 0, 'd'},
 {"precision", required_argument, 0, OPT_PRECISION},
 {NULL}
};
static const char *OptionsShort = "ho:r:s:i:c:d:p:";
static const char *OptionsDesc[] =
{
 ": prints help",
 ": specifies output deformation file instead of standart output",	
 ": specifies calculation region for grid, lon -180/180 or 0/360, lat -90/90",
 ": specifies file containing a set of points to calculate",
 ": sets grid spacial interval in ang min (1/1 by default)",
 ": specifies mu/(lambda+mu) ratio (0.5 by default)",
 ": adjusts fault centroid position (0: bottom left, 1: center [default], 2: bottom center, 3: top center)",
 ": DYNAMIC MODE -- generate displacement fields with specified time interval\n\r\tinput: FSP format, output: netCDF format",
 ": number of decimal digits of displacement to output (default is 5)",
 NULL
};
//===============================
string FFMFile = "", OutputFile = ""; // filenames
ifstream ffm;
ofstream out;
bool fSet = false, // set of points flag
 fDynamicMode = false; // dynamic mode flag
//===============================
void Usage(void)
{
 cout << "Finite fault crust surface displacement solver (v" << VERSION <<
#ifdef _OPENMP
  " w/ OpenMP support" <<
#endif
  ")" << endl << endl;
 cout << "Usage: ffaultdisp [FAULT] -r lon1/lat1/lon2/lat2 [-o dispfile] [-i lon_int/lat_int] [-c CR] [-p FP] [-d time_int] [--precision P]" << endl;
 cout << "  or:  ffaultdisp [FAULT] -s setfile [-o dispfile]  [-c CR] [-p FP] [--precision P]" << endl;
 cout << "Finite Fault Model data can be supplied via standart input or from file" << endl;
 cout << "Standart output will be used if -o option is omited" << endl << endl;
 
 cout << "Options:" << endl;
 for(int i=0;(const char*)OptionsLong[i].name != NULL;i++)
 {
  if(OptionsLong[i].val < OPT_PRECISION) cout << " -" << char(OptionsLong[i].val) << ",";
  cout << " --" << OptionsLong[i].name << OptionsDesc[i] << endl;
 }
}
//===============================
void CheckFileAccess(const char *fname)
{
 if(access(fname, R_OK) < 0)
 {
  cerr << "ERR: File \'" << fname << "\' ";
  switch(errno)
  {
   case ENOENT:
    cerr << "doesn't exists" << endl;
    break;

   case EACCES:
    cerr << "is not readable" << endl;
    break;

   default:
    cerr << "is not accessible" << endl;
  }
  exit(errno);
 }
}
//===============================
void ParseArgs(int argc, char *argv[])
{
int opt, option_index=0;
bool fRegion=false;
 try
 {
  while( (opt = getopt_long(argc, argv, OptionsShort, OptionsLong, &option_index)) != -1)
  {
   switch(opt)
   {
    case 'o':
     SetRes2Cout(false);
     OutputFile = optarg;
     break;

    case 'r':
     SetGridBounds(optarg);
     fRegion=true;
     break;

    case 's':
    {
     CheckFileAccess(optarg);
ifstream setstream(optarg, ios::in);
     ParseSet(setstream);
     fSet=true;
     break;
    }

    case 'i':
     SetGridInterval(optarg);
     break;

    case 'c':
     SetCR(optarg);
     break;

    case 'p':
     SetFaultPosition(optarg);
     break;

    case 'd':
     SetDynamicMode(optarg);
     fDynamicMode = true;
     break;

    case OPT_PRECISION:
     SetPrecision(optarg);
     break;

    case 'h':
    case '?':
     Usage();
     exit(0);
   }
  }
 }
 catch(char const* reason)
 {
  cerr << "ERR: " << reason << endl;
  exit(1);
 }

 if(optind < argc)  //  Getting [FAULT] 
 {
  FFMFile = argv[optind];
  CheckFileAccess(FFMFile.c_str());
 }

 if(!fSet && !fRegion) // FIXME
 {
  cerr << "ERR: Nor grid calculation region nor file with set of points aren't specified" << endl;
  exit(1);
 }
}
//===============================
int main(int argc, char *argv[])
{
#ifndef TEST_OKADA
 ParseArgs(argc, argv);

istream *ffmstream;
 if(!FFMFile.empty())
 {
  ffm.open(FFMFile.c_str(), ios::in);
  ffmstream = &ffm;
 } 
 else ffmstream = &cin;
 ParseFiniteFault(*ffmstream);


 if(fDynamicMode) // dynamic mode
 {
  if(OutputFile.empty())
  {
   cerr << "ERR: in-memory netCDF is not supported yet" << endl;
   exit(1);
  }
netCDF::NcFile nc(OutputFile.c_str(), netCDF::NcFile::replace);
  DynamicGrid(nc);
 }

 else // static mode
 {
ostream *outstream;
  if(!OutputFile.empty())
  {
   out.open(OutputFile.c_str(), ios::trunc);
   outstream = &out;
  }
  else outstream = &cout;

  if(!fSet)
  {
   StaticGrid();
   SaveGrid(*outstream);
  }
  else
  {
   StaticSet();
   SaveSet(*outstream);
  }
 }

#else
 TestOkada();
#endif
 return 0;
}