/*
 * Copyright (C) Sergey Kolesov 2012-2021 <kolesov@ocean.phys.msu.ru>
 * See ffaultfdisp.cpp for licensing details
 */

#include <cfloat>
#include "okada.h"
//===============================
// Defaults
TFaultPosition TFault::FaultPos = Middle;  //  Fault centroid position
double TFault::CR = 0.5; //  mu/(lambda+mu)
//===============================
TFault::TFault(double lon, double lat, double depth, double slip, double rake, double strike, double dip, double l, double w, double trup, double rise)
{
 Lon=lon; Lat=lat; Depth=depth; Slip=slip; Rake=rake; Strike=strike; Dip=dip; L=l; W=w;
 TRup = trup; Rise = rise;

 Rake *= deg2rad; Strike *= deg2rad; Dip *= deg2rad;
 //Slip /= 100;
 Depth *= 1000;

 cosDip = fabs(cos(Dip)) > DBL_EPSILON ? cos(Dip) : 0;
 sinDip = fabs(sin(Dip)) > DBL_EPSILON ? sin(Dip) : 0;
 cosRake = fabs(cos(Rake)) > DBL_EPSILON ? cos(Rake) : 0;
 sinRake = fabs(sin(Rake)) > DBL_EPSILON ? sin(Rake) : 0;
 cosStrike = fabs(cos(Strike)) > DBL_EPSILON ? cos(Strike) : 0;
 sinStrike = fabs(sin(Strike)) > DBL_EPSILON ? sin(Strike) : 0;
#ifndef TEST_OKADA
 if(FaultPos == Middle) Depth += W*sinDip/2;
#endif
}
//===============================
void TFault::OkadaFormulae(double x, double y, double& ux, double& uy, double& uz, double slip_frac)
{
const double U1 = -Slip*slip_frac*cosRake/(2*M_PI), U2 = -Slip*slip_frac*sinRake/(2*M_PI);
const double p = y*cosDip+Depth*sinDip, q = y*sinDip-Depth*cosDip;    //  From [eq.(30) p.1145]

  //  Displacement    [eq.(25) p.1144]
 ux = U1 * Chinnery(&TFault::ux_ss, x, p, q) + //  strike-slip
		U2 * Chinnery(&TFault::ux_ds, x, p, q);  //  dip-slip

 uy = U1 * Chinnery(&TFault::uy_ss, x, p, q) + //  strike-slip
		U2 * Chinnery(&TFault::uy_ds, x, p, q);  //  dip-slip

 uz = U1 * Chinnery(&TFault::uz_ss, x, p, q) + //  strike-slip
		U2 * Chinnery(&TFault::uz_ds, x, p, q);  //  dip-slip
}
//===============================
   //  Chinnery's notation [eq.(24) p.1143]
const double TFault::Chinnery(double (TFault::*F)(double, double, double), double x, double p, double q)
{
 return (*this.*F)(x, p, q) - (*this.*F)(x, p-W, q) - (*this.*F)(x-L, p, q) + (*this.*F)(x-L, p-W, q);
}
//===============================
   //  strike-slip displacement [eq.(25) p.1144]
double TFault::ux_ss(double xi, double eta, double q)
{
double R = sqrt(xi*xi + eta*eta + q*q);
 return ( fabs(R+eta)>DBL_EPSILON ? xi*q/(R*(R+eta)) : 0 ) + ( fabs(q)>DBL_EPSILON ? atan(xi*eta/(q*R)) : 0 ) + I1(xi, eta, q, R)*sinDip;
}
//===============================
double TFault::uy_ss(double xi, double eta, double q)
{
double R = sqrt(xi*xi + eta*eta + q*q);
 return ( fabs(R+eta)>DBL_EPSILON ? ((eta*cosDip + q*sinDip)*q + R*q*cosDip)/(R*(R+eta)) : 0 ) + I2(eta, q, R)*sinDip;
}
//===============================
double TFault::uz_ss(double xi, double eta, double q)
{
double R = sqrt(xi*xi + eta*eta + q*q);
double d_ = eta*sinDip - q*cosDip;
 return ( fabs(R+eta)>DBL_EPSILON ? (d_*q + R*q*sinDip)/(R*(R+eta)) : 0 ) + I4(d_, eta, q, R)*sinDip;
}
//===============================
   //  dip-slip displacement [eq.(26) p.1144]
double TFault::ux_ds(double xi, double eta, double q)
{
double R = sqrt(xi*xi + eta*eta + q*q);
 return q/R - I3(eta, q, R)*sinDip*cosDip;
}
//===============================
double TFault::uy_ds(double xi, double eta, double q)
{
double R = sqrt(xi*xi + eta*eta + q*q);
 return (eta*cosDip + q*sinDip)*q / (R*(R+xi)) + ( fabs(q)>DBL_EPSILON ? cosDip*atan(xi*eta/(q*R)) : 0 ) - I1(xi, eta, q, R)*sinDip*cosDip;
}
//===============================
double TFault::uz_ds(double xi, double eta, double q)
{
double R = sqrt(xi*xi + eta*eta + q*q);
double d_ = eta*sinDip - q*cosDip;
 return d_*q/(R*(R+xi)) + ( fabs(q)>DBL_EPSILON ? sinDip*atan(xi*eta/(q*R)) : 0 ) - I5(xi, eta, q, R, d_)*sinDip*cosDip;
}
//===============================
   // I terms [eq.(28)-(29) p.1144-1145]
double TFault::I1(double xi, double eta, double q, double R)
{
double d_ = eta*sinDip - q*cosDip;  //  d~ from [eq.(30) p.1145]
 if(fabs(cosDip)>DBL_EPSILON)
  return -CR*xi/(cosDip*(R+d_)) - sinDip/cosDip * I5(xi, eta, q, R, d_);

 return -CR/2*xi*q / ((R+d_)*(R+d_));
}
//===============================
double TFault::I2(double eta, double q, double R)
{
 return -CR*( R+eta>DBL_EPSILON ? log(R+eta) : -log(R-eta) ) - I3(eta, q, R);
}
//===============================
double TFault::I3(double eta, double q, double R)
{
double y_ = eta*cosDip + q*sinDip, d_ = eta*sinDip - q*cosDip;  //  y~ and d~ from [eq.(30) p.1145]
double log_ = R+eta>DBL_EPSILON ? log(R+eta) : -log(R-eta);
 if(fabs(cosDip)>DBL_EPSILON)
  return CR*(y_/(cosDip*(R+d_)) - log_) + sinDip/cosDip*I4(d_, eta, q, R);

 return CR/2*(eta/(R+d_) + y_*q/((R+d_)*(R+d_)) - log_);
}
//===============================
double TFault::I4(double d_, double eta, double q, double R)
{
 if(fabs(cosDip)>DBL_EPSILON)
  return CR/cosDip * (log(R+d_) - sinDip*( fabs(R+eta)>DBL_EPSILON ? log(R+eta) : -log(R-eta) ) );

 return  -CR*q/(R+d_);
}
//===============================
double TFault::I5(double xi, double eta, double q, double R, double d_)
{
 if(fabs(cosDip)>DBL_EPSILON)
 {
  if(fabs(xi) <= DBL_EPSILON) return 0;
double X = sqrt(xi*xi + q*q);
  return CR*2/cosDip * atan((eta*(X+q*cosDip) + X*(R+X)*sinDip)/(xi*(R+X)*cosDip));
 }

 return -CR*xi*sinDip/(R+d_);
}
//===============================
void TFault::CalcDisp(float lon, float lat, double& ue, double& un, double& uz, double slip_frac) // Static
{
const double YR = SphericalDist(Lon, Lat, Lon, lat) * (lat<Lat ? -1 : 1),
 R = SphericalDist(Lon, Lat, lon, lat),
 XR = SphericalDist(Lon, Lat, lon, Lat) * (lon<Lon ? -1 : 1),
 alpha = Strike - atan2(XR, YR);

double x = R*cos(alpha), y = R*sin(alpha);
 switch(FaultPos)
 {
  case Middle:
   x += L/2;
   y += W*cosDip/2;
   break;

  case BottomCenter:
   x += L/2;
   break;

  case TopCenter:
   x += L/2;
   y += W*cosDip;
   break;
 }

double ux, uy;
 OkadaFormulae(x, y, ux, uy, uz, slip_frac);
 // Rotating from local coordinates
 ue = ( sinStrike*ux - cosStrike*uy );
 un = ( cosStrike*ux + sinStrike*uy );
}
//===============================
void TFault::CalcDisp(float time, float time_inc, float lon, float lat, double& ue, double& un, double& uz) // Dynamic
{
float act_time = time_inc;
float prev_time = time-time_inc;
 if(prev_time < TRup) act_time = time-TRup;
 else if(time > TRup+Rise) act_time = TRup+Rise - prev_time;

double slip_frac = Rise != 0 ? act_time / Rise : 1; // Linear velocity assumption
 CalcDisp(lon, lat, ue, un, uz, slip_frac);
}
//===============================

//===============================
double SphericalDist(double Lon1, double Lat1, double Lon2, double Lat2)
{
 Lat1 *= deg2rad; Lat2 *= deg2rad;
 Lon1 *= deg2rad; Lon2 *= deg2rad;
 //double deltaX = (Lon2-Lon1) * RE * cos((Lat2+Lat1)/2), deltaY = (Lat2 - Lat1) * RE;

double dLon = Lon2-Lon1;
 if(fabs(dLon) > M_PI)
  dLon += (dLon > 0 ? -2*M_PI : 2*M_PI);
double deltaX = dLon * RE * cos((Lat2+Lat1)/2), deltaY = (Lat2-Lat1) * RE;
 return sqrt(deltaX*deltaX + deltaY*deltaY);
}
//===============================
#ifdef TEST_OKADA
#warning "TEST MODE ENABLED"
#include <iomanip>
#include <iostream>
void TestOkada(void)
{
/*
                    [Table 2 p.1149]
Okada test case 2
 strike (rake=0):   Ux=-8.689E-3, Uy=-4.298E-3, Uz=-2.747E-3
 dip (rake=90):     Ux=-4.682E-3, Uy=-3.527E-2, Uz=-3.564E-2

Okada test case 3
 strike (rake=0):   Ux=0, Uy=5.253E-3, Uz=0
 dip (rake=90):     Ux=0, Uy=0, Uz=0

Okada test case 4
 strike (rake=0):   Ux=0, Uy=-1.303E-3, Uz=0
 dip (rake=90):     Ux=0, Uy=0, Uz=0
*/

double ux, uy, uz;
// CR=0.5;

 cout << "Okada test case 2" << scientific << setprecision (3) << endl;
TFault case2s(2., 3., 0.004, 1., 0., 0., 70., 3., 2.);
 case2s.OkadaFormulae(2., 3., ux, uy, uz);
 cout << "strike: " << ux << '\t' << uy << '\t' << uz << endl;
TFault case2d(2., 3., 0.004, 1., 90., 0., 70., 3., 2.);
 case2d.OkadaFormulae(2., 3., ux, uy, uz);
 cout << "dip: \t" << ux << '\t' << uy << '\t' << uz << endl;

 cout << endl << "Okada test case 3" << endl;
TFault case3s(0., 0., 0.004, 1., 0., 0., 90., 3., 2.);
 case3s.OkadaFormulae(0., 0., ux, uy, uz);
 cout << "strike: " << ux << '\t' << uy << '\t' << uz << endl;
TFault case3d(0., 0., 0.004, 1., 90., 0., 90., 3., 2.);
 case3d.OkadaFormulae(0., 0., ux, uy, uz);
 cout << "dip: \t" << ux << '\t' << uy << '\t' << uz << endl;

 cout << endl << "Okada test case 4" << endl;
TFault case4s(0., 0., 0.004, 1., 0., 0., -90., 3., 2.);
 case4s.OkadaFormulae(0., 0., &ux, &uy, &uz);
 cout << "strike: " << ux << '\t' << uy << '\t' << uz << endl;
TFault case4d(0., 0., 0.004, 1., 90., 0., -90., 3., 2.);
 case4d.OkadaFormulae(0., 0., &ux, &uy, &uz);
 cout << "dip: \t" << ux << '\t' << uy << '\t' << uz << endl;
}
#endif
//===============================