/*
 * Copyright (C) Sergey Kolesov 2012-2021 <kolesov@ocean.phys.msu.ru>
 * See ffaultfdisp.cpp for licensing details
 */

#ifdef _OPENMP
#include <omp.h>
#endif
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <iomanip>
#include <algorithm>

#include "solver.h"
#include "nc.h"
//===============================
double LeftLon = 0, RightLon = 0, BottomLat = 0, TopLat = 0, LonStep = 1./60, LatStep = 1./60;   //  Grid region & intervals
unsigned Nlon, Nlat; // Grid dimensions
vector<TFault> FaultPlane;   //  matrix of subfaults
vector<double> DispX, DispY, DispZ; //  bottom displacement
vector<float> Lons, Lats; // set of points' coordinates
vector<string> Comments; // comments on points
bool fRes2Cout = true; // results to stdout flag
int Precision = 5; // number of decimal digits of displacement for output
float DynamicTimeStep;
string Event = "", EventTAG = "";
//===============================
double LonTo360(double l)
{
int revols = static_cast<int>(l / 360) * 360;
 return l < 0 ? l + revols + 360 : l - revols;
}
//===============================
void CheckLon(double& Lon1, double& Lon2)
{
 Lon1 = LonTo360(Lon1);
 Lon2 = LonTo360(Lon2);
 if(Lon1 == Lon2) throw "Equal longitudes in a region";
 if(Lon1 > Lon2) Lon2 += 360;
 if(Lon2 - Lon1 > 180) cout << "WRN: Huge longitude span: " << Lon2 - Lon1 << " deg"<< endl; // Warn if longitude span is more than 180 deg
}
//===============================
void CheckLat(double& Lat1, double& Lat2)
{
 if(Lat1 == Lat2) throw "Equal latitudes in a region";
 if(Lat1 < -90 || Lat2 < -90 || Lat1 > 90 || Lat2 > 90) throw "Latitude is definitely wrong";
 if(Lat1 > Lat2) swap(Lat1, Lat2);
 if(Lat2 - Lat1 > 90)  cout << "WRN: Huge latitude span: " << Lat2 - Lat1 << " deg" << endl; // Warn if latitude span is more than 90 deg
}
//===============================
void SetGridBounds(char* Str)
{
stringstream str(Str);
 str >> LeftLon;
 str.ignore(255, '/');
 str >> BottomLat;
 str.ignore(255, '/');
 str >> RightLon;
 str.ignore(255, '/');
 str >> TopLat;
 if(str.fail()) throw "Syntax error in region specification";
 CheckLon(LeftLon, RightLon);
 CheckLat(BottomLat, TopLat);
}
//===============================
void SetGridInterval(char* Str)
{
stringstream str(Str);
 str >> LonStep;
 str.ignore(255, '/');
 str >> LatStep;
 if(str.fail()) throw "Syntax error in grid interval specification";
 if(LonStep <= 0 || LonStep > 60 || LatStep <= 0 || LatStep > 60) throw "Grid interval is negative or more than 1 deg"; //  More than 1 deg step is treated as typo

 LonStep /= 60; LatStep /= 60;
}
//===============================
void SetRes2Cout(bool O_RLY)
{
 fRes2Cout = O_RLY;
}
//===============================
void SetCR(char* Str)
{
stringstream str(Str);
 str >> TFault::CR;
 if(str.fail()) throw "Wrong CR value specified";
 if(TFault::CR <= 0 || TFault::CR > 10) throw "CR is zero, negative or exceeds 10";   //  Sane CR interval
}
//===============================
void SetFaultPosition(char* Str)
{
stringstream str(Str);
int fp;
 str >> fp;
 if(str.fail()) throw "Syntax error in fault centroid position specification";
 if(fp < 0 || fp > 3)  throw "Fault centroid position should be from 0 to 3 (see help)";

 TFault::FaultPos = TFaultPosition(fp);
}
//===============================
void SetDynamicMode(char* Str)
{
stringstream str(Str);
 str >> DynamicTimeStep;
 if(str.fail()) throw "Wrong time interval";
}
//===============================
void SetPrecision(char* Str)
{
stringstream str(Str);
 str >> Precision;
 if(str.fail()) throw "Wrong output precision value";
 if(Precision < 0)  throw "Output precision should be positive";
}
//===============================
/* Own format:

Lat Lon depth[km] slip[cm] rake strike dip L[m] W[m]
*/

void OwnFormat(istream& Is, int& ln)
{
string line = "";
 do
 {
  ln++;
  getline(Is, line);

int c = line.find_first_of("#\r");   // Eliminating comments and CRs
  if(c != string::npos) line=line.substr(0, c);
  c = line.find_first_not_of(" \t"); // Skip empty strings
  if(c == string::npos) continue;

  CreateFault(line); // Creating fault from string
 } while(1);
}
//===============================
/* Fsp (_f_inite-_s_ource _p_arameter) format:

http://equake-rc.info/SRCMOD/fileformats/fsp/
*/
void FspGetPair(string line, double& a, double& b)
{
auto sl = stringstream(line);
 sl.ignore(255, '=');
 sl >> a;
 sl.ignore(255, '=');
 sl >> b;
}
//===============================
void FspGetString(string line, string& s)
{
size_t pos = line.find(':') + 2; // Eliminate extra space
 s = line.substr(pos);
}
//===============================
void FspFormat(istream& Is, int& ln)
{
string line = "";
const char *wl_line_begin = "% Invs : Dx",
 *mech_line_begin = "% Mech",
 *seg_line_begin = "% SEGMENT",
 *event_line_begin = "% Event ",
 *eventtag_line_begin = "% EventTAG";
double w, l, strk, dip;

 do
 {
  ln++;
  getline(Is, line);

  if(line.find(event_line_begin) != string::npos)
   FspGetString(line, Event);

  if(line.find(eventtag_line_begin) != string::npos)
   FspGetString(line, EventTAG);

 if(line.find(mech_line_begin) != string::npos)
   FspGetPair(line, strk, dip);

 if(line.find(seg_line_begin) != string::npos)
   FspGetPair(line, strk, dip);

  if(line.find(wl_line_begin) != string::npos)
   FspGetPair(line, l, w);

  if(line[0] != '%')
   CreateFault(line, l*1000, w*1000, strk, dip);
 } while(1);
}
//===============================
void ParseFiniteFault(istream& Is)
{
const char *FSP_MAGIC = "% ----";
int ln = 0;
string firstline = "";
stringstream sis;
 try
 {
  Is.exceptions(istream::failbit | istream::badbit);
  sis.exceptions(istream::failbit | istream::badbit);
  copy(istreambuf_iterator<char>(Is), istreambuf_iterator<char>(), ostreambuf_iterator<char>(sis));

  getline(sis, firstline);
  if(firstline.substr(0, ((string)FSP_MAGIC).size()) == (string)FSP_MAGIC) FspFormat(sis.seekg(0), ln);
  else OwnFormat(sis.seekg(0), ln);
 }
 catch(string& what)
 {
  cerr << "ERR: Failed to parse input data:\"" << what << "\" (line " << ln << ")" << endl;
  exit(1);
 }
// catch(istream::failure& e)
 catch(...)
 {
  if(sis.eof()) return;
  cerr << "ERR: Something went wrong while reading input (line " << ln << ")" << endl;
  exit(1);
 }
}
//===============================
void CreateFault(string& Str, double l, double w, double strk, double dip)
{
stringstream stream(Str);
double Lat, Lon, Depth, Slip, Rake, Strike=strk, Dip=dip, L=l, W=w, TRup, Rise, _tmp;

 if(L == 0)
 {
  stream >> Lat >> Lon >> Depth >> Slip >> Rake >> Strike >> Dip >> L >> W;
  Slip /= 100;
TFault fault(LonTo360(Lon), Lat, Depth, Slip, Rake, Strike, Dip, L, W);
  FaultPlane.push_back(fault);
  FaultPlane.back().id = FaultPlane.size();
 }
 else
 { // fsp
  stream >> Lat >> Lon >> _tmp >> _tmp >> Depth >> Slip >> Rake >> TRup >> Rise >> _tmp;
TFault fault(LonTo360(Lon), Lat, Depth, Slip, Rake, Strike, Dip, L, W, TRup, Rise);
  FaultPlane.push_back(fault);
  FaultPlane.back().id = FaultPlane.size();
 }
 if(stream.fail()) throw Str;
}
//===============================
void StaticGrid(void)
{
 Nlon = floor((RightLon-LeftLon)/LonStep+0.5) + 1;
 Nlat = floor((TopLat-BottomLat)/LatStep+0.5) + 1;

 DispX.resize(Nlon*Nlat, 0);
 DispY.resize(Nlon*Nlat, 0);
 DispZ.resize(Nlon*Nlat, 0);

 for(int f=0;f<FaultPlane.size();f++)
 {
int nlat;
#pragma omp parallel for shared(nlat, DispX, DispY, DispZ)
  for(nlat=0;nlat<Nlat;nlat++)
  {
const float lat = BottomLat+nlat*LatStep;
   for(int nlon=0;nlon<Nlon;nlon++)
   {
const float lon = LeftLon+nlon*LonStep;
double ue, un, uz;
    FaultPlane[f].CalcDisp(lon, lat, ue, un, uz);

    DispX[nlat*Nlon+nlon] += ue;
    DispY[nlat*Nlon+nlon] += un;
    DispZ[nlat*Nlon+nlon] += uz;
   }
  }

  if(!fRes2Cout) cout << "Fault " << f+1 << " of " << FaultPlane.size() << '\r' << flush;
 }
 if(!fRes2Cout) cout << endl;
}
//===============================
void SaveGrid(ostream& Os)
{
 for(int nlat=0;nlat<Nlat;nlat++)
  for(int nlon=0;nlon<Nlon;nlon++)
  {
   Os.unsetf(ios_base::fixed);
   Os << setprecision(8) << LeftLon+nlon*LonStep << '\t' << BottomLat+nlat*LatStep << '\t';
   Os.setf(ios_base::fixed);
   Os << setprecision(Precision) << DispX[nlat*Nlon+nlon] << '\t' << DispY[nlat*Nlon+nlon] << '\t' << DispZ[nlat*Nlon+nlon] << '\n';
  }
}
//===============================
void ParseSet(istream& Is)
{
string line = "";
int ln = 0;
double lon, lat;
string comment;

 Is >> setw(255);   //   Get 255 chars at max
 do
 {
  ln++;
  getline(Is, line);
  if(Is.eof()) break;
stringstream Str(line);
  /*
   Points set file format:
   Lon  Lat  Comment-till-the-end-of-line
  */
  Str >> lon >> lat;
  getline(Str, comment);
  if(Str.fail()) cerr << "WRN: Failed to parse point:\"" << line << "\" (line " << ln << ")" << endl;
  else
  {
   Lons.push_back(lon);
   Lats.push_back(lat);
   Comments.push_back(comment);
  }
 } while(1);
}
//===============================
void StaticSet(void)
{
 DispX.resize(Lons.size(), 0);
 DispY.resize(Lons.size(), 0);
 DispZ.resize(Lons.size(), 0);

 for(int i=0;i<Lons.size();i++)
 {
int f;
#pragma omp parallel for shared(f, DispX, DispY, DispZ)
  for(f=0;f<FaultPlane.size();f++)
  {
double ue, un, uz;
   FaultPlane[f].CalcDisp(Lons[i], Lats[i], ue, un, uz);
#pragma omp atomic
    DispX[i] += ue;
#pragma omp atomic
    DispY[i] += un;
#pragma omp atomic
    DispZ[i] += uz;
  }

  if(!fRes2Cout) cout << "Point " << i+1 << " of " << Lons.size() << '\r' << flush;
 }
 if(!fRes2Cout) cout << endl;
}
//===============================
void SaveSet(ostream& Os)
{
 for(int i=0;i<Lons.size();i++)
 {
  Os.unsetf(ios_base::fixed);
  Os << setprecision(8) << Lons[i] << '\t' << Lats[i] << '\t';
  Os.setf(ios_base::fixed);
  Os << setprecision(Precision) << DispX[i] << '\t' << DispY[i] << '\t' << DispZ[i] << '\t' << Comments[i] << '\n';
 }
}
//===============================
void DynamicGrid(netCDF::NcFile& nc)
{
 Nlon = floor((RightLon-LeftLon)/LonStep+0.5) + 1;
 Nlat = floor((TopLat-BottomLat)/LatStep+0.5) + 1;

vector<float> lons(Nlon);
vector<float> lats(Nlat);
 for(int nlat=0;nlat<Nlat;nlat++)
  lats[nlat] = BottomLat+nlat*LatStep;
 for(int nlon=0;nlon<Nlon;nlon++)
  lons[nlon] = LeftLon+nlon*LonStep;

 NetCDFInit(nc, Nlon, Nlat, lons.data(), lats.data(), DynamicTimeStep, Event, EventTAG);

 DispX.resize(Nlon*Nlat, 0);
 DispY.resize(Nlon*Nlat, 0);
 DispZ.resize(Nlon*Nlat, 0);

unsigned idx_time = 0;
 while(!FaultPlane.empty())
 {
const float time = idx_time*DynamicTimeStep;
  for(auto f=FaultPlane.begin();f!=FaultPlane.end();++f)
  {
   if(!fRes2Cout) cout << "\33[2K\rTime: " << time << "s, fault " << (f+1-FaultPlane.begin()) << " of " << FaultPlane.size() << flush;
   if((*f).BeforeRupture(time)) continue;
int nlat;
#pragma omp parallel for shared(nlat, DispX, DispY, DispZ)
   for(nlat=0;nlat<Nlat;nlat++)
   {
const double lat = BottomLat+nlat*LatStep;
    for(int nlon=0;nlon<Nlon;nlon++)
    {
const double lon = LeftLon+nlon*LonStep;
double ue, un, uz;
     (*f).CalcDisp(time, DynamicTimeStep, lon, lat, ue, un, uz);

     DispX[nlat*Nlon+nlon] += ue;
     DispY[nlat*Nlon+nlon] += un;
     DispZ[nlat*Nlon+nlon] += uz;
    }
   }
  }

  // remove terminated subfaults
auto from = remove_if(FaultPlane.begin(), FaultPlane.end(), [&time](TFault ff) { return ff.AfterRupture(time); });
  FaultPlane.erase(from, FaultPlane.end());

  SaveTimeSlice(nc, idx_time, Nlon, Nlat, DispX.data(), DispY.data(), DispZ.data());
  idx_time++;
 }

 if(!fRes2Cout) cout << "\33[2K\rSource duration: " << (idx_time-1)*DynamicTimeStep << " second" << endl;
}

